<?php
require_once __DIR__ . '/_auth.php';
require_once __DIR__ . '/../src/db.php';

// Only owner / super_admin can manage users
require_role(['owner', 'super_admin']);

$pageTitle  = 'Team & Access';
$activeMenu = 'admins';

$msg = '';
$err = '';

$rolesAvailable = [
    'owner'         => 'Owner (full access)',
    'super_admin'   => 'Super Admin',
    'admin'         => 'Admin (almost full)',
    'event_manager' => 'Event Manager',
    'finance'       => 'Finance',
    'gate'          => 'Gate Operator',
    'viewer'        => 'Viewer (read-only)',
];

// Handle create / update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'create') {
        $name     = trim($_POST['name'] ?? '');
        $username = trim($_POST['username'] ?? '');
        $role     = $_POST['role'] ?? 'admin';
        $password = $_POST['password'] ?? '';

        if ($name === '' || $username === '' || $password === '') {
            $err = 'Name, username and password are required.';
        } elseif (!isset($rolesAvailable[$role])) {
            $err = 'Invalid role selected.';
        } else {
            // Check duplicate username
            $check = $pdo->prepare("SELECT id FROM admins WHERE username = ? LIMIT 1");
            $check->execute([$username]);
            if ($check->fetch()) {
                $err = 'Username already exists.';
            } else {
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("
                    INSERT INTO admins (name, username, password_hash, role, is_active)
                    VALUES (?, ?, ?, ?, 1)
                ");
                $stmt->execute([$name, $username, $hash, $role]);
                $msg = 'Admin user created.';
            }
        }
    } elseif ($action === 'update') {
        $id       = (int)($_POST['id'] ?? 0);
        $name     = trim($_POST['name'] ?? '');
        $username = trim($_POST['username'] ?? '');
        $role     = $_POST['role'] ?? 'admin';
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $password = $_POST['password'] ?? '';

        if ($id <= 0) {
            $err = 'Invalid admin ID.';
        } elseif (!isset($rolesAvailable[$role])) {
            $err = 'Invalid role.';
        } elseif ($name === '' || $username === '') {
            $err = 'Name and username are required.';
        } else {
            // Do not allow owner to deactivate themselves (basic safety)
            if ($id === (int)$ADMIN_ID && $isActive === 0) {
                $err = 'You cannot deactivate yourself.';
            } else {
                // Ensure username is unique for others
                $check = $pdo->prepare("SELECT id FROM admins WHERE username = ? AND id <> ? LIMIT 1");
                $check->execute([$username, $id]);
                if ($check->fetch()) {
                    $err = 'Username already taken by another user.';
                } else {
                    if ($password !== '') {
                        $hash = password_hash($password, PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("
                            UPDATE admins
                            SET name = ?, username = ?, role = ?, is_active = ?, password_hash = ?
                            WHERE id = ?
                        ");
                        $stmt->execute([$name, $username, $role, $isActive, $hash, $id]);
                    } else {
                        $stmt = $pdo->prepare("
                            UPDATE admins
                            SET name = ?, username = ?, role = ?, is_active = ?
                            WHERE id = ?
                        ");
                        $stmt->execute([$name, $username, $role, $isActive, $id]);
                    }
                    $msg = 'Admin updated.';
                }
            }
        }
    }
}

// Load current admins
$admins = $pdo->query("
    SELECT id, name, username, role, is_active, created_at
    FROM admins
    ORDER BY id ASC
")->fetchAll();

include __DIR__ . '/_layout_top.php';
?>

<div>
    <h1 class="page-heading">Team & Access</h1>
    <p class="page-subtitle">
        Manage who can access TUMESSE EVENTS and what each person is allowed to do.
    </p>
</div>

<div class="card" style="max-width:780px;margin-bottom:16px;">
    <?php if ($msg): ?>
        <div style="background:#e0f2fe;border:1px solid #bfdbfe;color:#1d4ed8;padding:8px 10px;border-radius:8px;font-size:.85rem;margin-bottom:12px;">
            <?= htmlspecialchars($msg) ?>
        </div>
    <?php endif; ?>
    <?php if ($err): ?>
        <div style="background:#fee2e2;border:1px solid #fecaca;color:#b91c1c;padding:8px 10px;border-radius:8px;font-size:.85rem;margin-bottom:12px;">
            <?= htmlspecialchars($err) ?>
        </div>
    <?php endif; ?>

    <h3 style="margin-top:0;margin-bottom:8px;font-size:1rem;">Add new team member</h3>
    <form method="post" style="display:flex;flex-direction:column;gap:8px;">
        <input type="hidden" name="action" value="create">
        <div class="grid" style="grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:10px;">
            <div>
                <label style="font-size:.85rem;">Full name</label><br>
                <input type="text" name="name" required
                       style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
            </div>
            <div>
                <label style="font-size:.85rem;">Username</label><br>
                <input type="text" name="username" required
                       style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
            </div>
            <div>
                <label style="font-size:.85rem;">Role</label><br>
                <select name="role"
                        style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
                    <?php foreach ($rolesAvailable as $key => $label): ?>
                        <option value="<?= htmlspecialchars($key) ?>">
                            <?= htmlspecialchars($label) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        <div>
            <label style="font-size:.85rem;">Password</label><br>
            <input type="password" name="password" required
                   style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
        </div>
        <div>
            <button class="btn-action primary" type="submit">Create user</button>
        </div>
    </form>
</div>

<div class="card">
    <h3 style="margin-top:0;margin-bottom:8px;font-size:1rem;">Existing users</h3>
    <?php if (!$admins): ?>
        <p style="font-size:.86rem;color:#6b7280;">No admins found.</p>
    <?php else: ?>
        <div style="overflow-x:auto;">
            <table style="width:100%;border-collapse:collapse;font-size:.86rem;">
                <thead>
                <tr>
                    <th style="padding:7px 8px;border-bottom:1px solid #e5e7eb;text-align:left;">ID</th>
                    <th style="padding:7px 8px;border-bottom:1px solid #e5e7eb;text-align:left;">User</th>
                    <th style="padding:7px 8px;border-bottom:1px solid #e5e7eb;text-align:left;">Role</th>
                    <th style="padding:7px 8px;border-bottom:1px solid #e5e7eb;text-align:left;">Status</th>
                    <th style="padding:7px 8px;border-bottom:1px solid #e5e7eb;text-align:left;">Actions</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($admins as $a): ?>
                    <tr>
                        <td style="padding:7px 8px;border-bottom:1px solid #e5e7eb;"><?= (int)$a['id'] ?></td>
                        <td style="padding:7px 8px;border-bottom:1px solid #e5e7eb;">
                            <strong><?= htmlspecialchars($a['name'] ?: $a['username']) ?></strong><br>
                            <span class="pill-label">Username: <?= htmlspecialchars($a['username']) ?></span>
                        </td>
                        <td style="padding:7px 8px;border-bottom:1px solid #e5e7eb;">
                            <?= htmlspecialchars($a['role']) ?>
                        </td>
                        <td style="padding:7px 8px;border-bottom:1px solid #e5e7eb;">
                            <?php if ($a['is_active']): ?>
                                <span class="badge badge-paid">Active</span>
                            <?php else: ?>
                                <span class="badge badge-unpaid">Disabled</span>
                            <?php endif; ?>
                        </td>
                        <td style="padding:7px 8px;border-bottom:1px solid #e5e7eb;">
                            <details>
                                <summary style="cursor:pointer;font-size:.82rem;color:#2563eb;">Edit</summary>
                                <form method="post" style="margin-top:6px;font-size:.82rem;display:flex;flex-direction:column;gap:6px;">
                                    <input type="hidden" name="action" value="update">
                                    <input type="hidden" name="id" value="<?= (int)$a['id'] ?>">

                                    <div class="grid" style="grid-template-columns:repeat(auto-fit,minmax(160px,1fr));gap:6px;">
                                        <div>
                                            <label>Name</label><br>
                                            <input type="text" name="name"
                                                   value="<?= htmlspecialchars($a['name']) ?>"
                                                   style="width:100%;padding:5px 7px;border-radius:6px;border:1px solid #d5d8e5;">
                                        </div>
                                        <div>
                                            <label>Username</label><br>
                                            <input type="text" name="username"
                                                   value="<?= htmlspecialchars($a['username']) ?>"
                                                   style="width:100%;padding:5px 7px;border-radius:6px;border:1px solid #d5d8e5;">
                                        </div>
                                        <div>
                                            <label>Role</label><br>
                                            <select name="role"
                                                    style="width:100%;padding:5px 7px;border-radius:6px;border:1px solid #d5d8e5;">
                                                <?php foreach ($rolesAvailable as $key => $label): ?>
                                                    <option value="<?= htmlspecialchars($key) ?>" <?= $a['role'] === $key ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($label) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>

                                    <div>
                                        <label style="font-size:.8rem;">
                                            <input type="checkbox" name="is_active" value="1" <?= $a['is_active'] ? 'checked' : '' ?>>
                                            Active
                                        </label>
                                        <div style="font-size:.75rem;color:#6b7280;">
                                            Uncheck to temporarily disable login for this user.
                                        </div>
                                    </div>

                                    <div>
                                        <label>New password (optional)</label><br>
                                        <input type="password" name="password"
                                               placeholder="Leave blank to keep current"
                                               style="width:100%;padding:5px 7px;border-radius:6px;border:1px solid #d5d8e5;">
                                    </div>

                                    <div style="display:flex;gap:6px;">
                                        <button class="btn-action primary" type="submit">Save</button>
                                    </div>
                                </form>
                            </details>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php include __DIR__ . '/_layout_bottom.php'; ?>