<?php
require_once __DIR__ . '/_auth.php';
require_once __DIR__ . '/../src/db.php';
require_once __DIR__ . '/../src/events.php';

$pageTitle  = 'Dashboard';
$activeMenu = 'dashboard';

$currentEventId = getCurrentEventId();
$events         = getAllEvents();
$currentEvent   = null;
foreach ($events as $ev) {
    if ((int)$ev['id'] === $currentEventId) {
        $currentEvent = $ev;
        break;
    }
}

$adminRole = $_SESSION['admin_role'] ?? 'admin';

// --- SUMMARY STATS (users table) ---------------------------------------------
$stmt = $pdo->prepare("
    SELECT
        COUNT(*) AS guests,
        COALESCE(SUM(people_count),0) AS people,
        COALESCE(SUM(COALESCE(price_total,0)),0) AS amount_total,
        COALESCE(SUM(LEAST(COALESCE(amount_paid,0), COALESCE(price_total,0))),0) AS amount_paid,
        COALESCE(SUM(GREATEST(COALESCE(price_total,0) - COALESCE(amount_paid,0), 0)),0) AS amount_due,
        COALESCE(SUM(CASE
            WHEN (price_total IS NOT NULL AND price_total > 0 AND amount_paid >= price_total)
                 OR paid = 1
            THEN 1 ELSE 0 END),0) AS count_paid,
        COALESCE(SUM(CASE
            WHEN price_total IS NOT NULL AND price_total > 0
                 AND amount_paid > 0 AND amount_paid < price_total
            THEN 1 ELSE 0 END),0) AS count_partial,
        COALESCE(SUM(CASE
            WHEN (price_total IS NOT NULL AND (amount_paid = 0 OR amount_paid IS NULL))
                 OR paid = 0
            THEN 1 ELSE 0 END),0) AS count_unpaid,
        COALESCE(SUM(CASE WHEN used = 1 THEN 1 ELSE 0 END),0) AS count_checked
    FROM users
    WHERE event_id = ?
");
$stmt->execute([$currentEventId]);
$summary = $stmt->fetch() ?: [
    'guests' => 0, 'people' => 0, 'amount_total' => 0,
    'amount_paid' => 0, 'amount_due' => 0,
    'count_paid' => 0, 'count_partial' => 0, 'count_unpaid' => 0,
    'count_checked' => 0,
];

// --- SMS ACTIVITY (sends table) ---------------------------------------------
$stmtSms = $pdo->prepare("
    SELECT
        COUNT(*) AS total_sends,
        SUM(CASE WHEN created_at >= (NOW() - INTERVAL 1 DAY) THEN 1 ELSE 0 END) AS sends_24h
    FROM sends
");
$stmtSms->execute();
$smsStats = $stmtSms->fetch() ?: ['total_sends' => 0, 'sends_24h' => 0];

// --- LATEST GUESTS -----------------------------------------------------------
$stmtGuests = $pdo->prepare("
    SELECT *
    FROM users
    WHERE event_id = ?
    ORDER BY id DESC
    LIMIT 6
");
$stmtGuests->execute([$currentEventId]);
$latestGuests = $stmtGuests->fetchAll();

// --- LATEST RSVPS ------------------------------------------------------------
$stmtRsvp = $pdo->prepare("
    SELECT *
    FROM rsvps
    WHERE event_id = ?
    ORDER BY id DESC
    LIMIT 6
");
$stmtRsvp->execute([$currentEventId]);
$latestRsvps = $stmtRsvp->fetchAll();

// --- LATEST CHECK-INS ---------------------------------------------------------
$stmtCheckIns = $pdo->prepare("
    SELECT id, name, phone, used_at
    FROM users
    WHERE event_id = ? AND used = 1 AND used_at IS NOT NULL
    ORDER BY used_at DESC
    LIMIT 6
");
$stmtCheckIns->execute([$currentEventId]);
$checkIns = $stmtCheckIns->fetchAll();

include __DIR__ . '/_layout_top.php';
?>

<div>
    <h1 class="page-heading">Overview</h1>
    <p class="page-subtitle">
        Quick snapshot for
        <strong><?= htmlspecialchars($currentEvent['name'] ?? 'Current event') ?></strong>
        <?php if (!empty($currentEvent['event_date'])): ?>
            · <?= htmlspecialchars($currentEvent['event_date']) ?>
        <?php endif; ?>
        <?php if (!empty($currentEvent['venue'])): ?>
            · <?= htmlspecialchars($currentEvent['venue']) ?>
        <?php endif; ?>
    </p>
</div>

<div class="grid">
    <div class="card card-accent-blue">
        <div class="card-title">Guests &amp; capacity</div>
        <div class="card-value"><?= (int)$summary['guests'] ?></div>
        <div class="card-meta">
            People capacity: <strong><?= (int)$summary['people'] ?></strong>
        </div>
    </div>

    <div class="card card-accent-green">
        <div class="card-title">Payments (TZS)</div>
        <div class="card-value"><?= number_format($summary['amount_total'], 0) ?></div>
        <div class="card-meta">
            Paid: <strong><?= number_format($summary['amount_paid'], 0) ?></strong>
            · Due: <strong><?= number_format($summary['amount_due'], 0) ?></strong>
        </div>
    </div>

    <div class="card card-accent-orange">
        <div class="card-title">Payment status</div>
        <div class="card-meta">
            <span class="badge badge-paid"><?= (int)$summary['count_paid'] ?> Paid</span>
            <span class="badge badge-unpaid"><?= (int)$summary['count_partial'] ?> Partial</span>
            <span class="badge badge-notused"><?= (int)$summary['count_unpaid'] ?> Unpaid</span>
        </div>
        <div class="card-meta" style="margin-top:6px;">
            Check-ins: <span class="badge badge-used"><?= (int)$summary['count_checked'] ?> scanned</span>
        </div>
    </div>

    <div class="card card-accent-purple">
        <div class="card-title">Messaging</div>
        <div class="card-value"><?= (int)$smsStats['total_sends'] ?></div>
        <div class="card-meta">
            SMS sent in last 24h: <strong><?= (int)$smsStats['sends_24h'] ?></strong>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-title">Quick actions</div>
    <div class="actions-row" style="margin-top:6px;">
        <a href="dashboard.php" class="btn-action">Refresh overview</a>

        <?php if (in_array($adminRole, ['owner','super_admin','admin','event_manager'], true)): ?>
            <a href="add_guest.php" class="btn-action primary">Add guest</a>
            <a href="guests.php" class="btn-action">View guests &amp; tickets</a>
            <a href="send_bulk.php" class="btn-action">Send bulk SMS</a>
        <?php endif; ?>

        <a href="../public/scanner.html" target="_blank" class="btn-action">Open gate scanner</a>
        <a href="reports.php" class="btn-action">View reports</a>
    </div>
</div>

<div class="grid">
    <div class="card">
        <div class="card-title">Latest guests</div>
        <?php if (!$latestGuests): ?>
            <div class="card-meta">No guests added yet.</div>
        <?php else: ?>
            <div style="overflow-x:auto;margin-top:6px;">
                <table>
                    <thead>
                    <tr>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Ticket</th>
                        <th>Paid</th>
                        <th>Created</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($latestGuests as $g): ?>
                        <?php
                        $priceTotal = $g['price_total'] !== null ? (float)$g['price_total'] : 0;
                        $paid       = (float)($g['amount_paid'] ?? 0);
                        $status     = 'Unpaid';
                        $badgeClass = 'badge-unpaid';
                        if ($priceTotal > 0) {
                            if ($paid >= $priceTotal) {
                                $status     = 'Paid';
                                $badgeClass = 'badge-paid';
                            } elseif ($paid > 0) {
                                $status     = 'Partial';
                                $badgeClass = 'badge-unpaid';
                            }
                        } elseif (!empty($g['paid'])) {
                            $status     = 'Paid';
                            $badgeClass = 'badge-paid';
                        }
                        ?>
                        <tr>
                            <td><?= htmlspecialchars($g['name']) ?></td>
                            <td><?= htmlspecialchars($g['phone']) ?></td>
                            <td><?= htmlspecialchars(ucfirst($g['pricing'] ?? '')) ?></td>
                            <td><span class="badge <?= $badgeClass ?>"><?= $status ?></span></td>
                            <td><?= htmlspecialchars($g['created_at']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <div class="card">
        <div class="card-title">Recent RSVPs</div>
        <?php if (!$latestRsvps): ?>
            <div class="card-meta">No RSVPs yet for this event.</div>
        <?php else: ?>
            <div style="overflow-x:auto;margin-top:6px;">
                <table>
                    <thead>
                    <tr>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Message</th>
                        <th>Time</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($latestRsvps as $r): ?>
                        <tr>
                            <td><?= htmlspecialchars($r['name']) ?></td>
                            <td><?= htmlspecialchars($r['phone']) ?></td>
                            <td style="max-width:220px;">
                                <div style="max-height:56px;overflow:auto;white-space:pre-wrap;">
                                    <?= htmlspecialchars($r['message']) ?>
                                </div>
                            </td>
                            <td><?= htmlspecialchars($r['created_at']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<div class="grid">
    <div class="card">
        <div class="card-title">Recent check-ins</div>
        <?php if (!$checkIns): ?>
            <div class="card-meta">No tickets scanned yet.</div>
        <?php else: ?>
            <div style="overflow-x:auto;margin-top:6px;">
                <table>
                    <thead>
                    <tr>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Checked in at</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($checkIns as $c): ?>
                        <tr>
                            <td><?= htmlspecialchars($c['name']) ?></td>
                            <td><?= htmlspecialchars($c['phone']) ?></td>
                            <td><?= htmlspecialchars($c['used_at']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <div class="card">
        <div class="card-title">System snapshot</div>
        <div class="card-meta" style="margin-bottom:4px;">
            <strong>Role:</strong> <?= htmlspecialchars($adminRole) ?>
        </div>
        <div class="card-meta" style="margin-bottom:4px;">
            <strong>Total events:</strong> <?= count($events) ?>
        </div>
        <div class="card-meta">
            Use the event switcher in the top-right to jump between different
            ceremonies and view their guests, payments and reports.
        </div>
    </div>
</div>

<?php include __DIR__ . '/_layout_bottom.php'; ?>