<?php
require_once __DIR__ . '/_auth.php';
require_once __DIR__ . '/../src/db.php';
require_once __DIR__ . '/../src/events.php';

$pageTitle  = 'Reports & Exports';
$activeMenu = 'reports';

// Current event (using same helper/session logic)
$currentEventId = getCurrentEventId();
$events         = getAllEvents();
$currentEvent   = null;
foreach ($events as $ev) {
    if ((int)$ev['id'] === $currentEventId) {
        $currentEvent = $ev;
        break;
    }
}

// --- MAIN SUMMARY STATS ---
$stmt = $pdo->prepare("
    SELECT
        COUNT(*) AS guests,
        COALESCE(SUM(people_count),0) AS people,
        COALESCE(SUM(COALESCE(price_total,0)),0) AS amount_total,
        COALESCE(SUM(LEAST(COALESCE(amount_paid,0), COALESCE(price_total,0))),0) AS amount_paid,
        COALESCE(SUM(GREATEST(COALESCE(price_total,0) - COALESCE(amount_paid,0), 0)),0) AS amount_due,
        COALESCE(SUM(CASE
            WHEN (price_total IS NOT NULL AND price_total > 0 AND amount_paid >= price_total)
                 OR paid = 1
            THEN 1 ELSE 0 END),0) AS count_paid,
        COALESCE(SUM(CASE
            WHEN price_total IS NOT NULL AND price_total > 0
                 AND amount_paid > 0 AND amount_paid < price_total
            THEN 1 ELSE 0 END),0) AS count_partial,
        COALESCE(SUM(CASE
            WHEN (price_total IS NOT NULL AND (amount_paid = 0 OR amount_paid IS NULL))
                 OR paid = 0
            THEN 1 ELSE 0 END),0) AS count_unpaid,
        COALESCE(SUM(CASE WHEN used = 1 THEN 1 ELSE 0 END),0) AS count_checked
    FROM users
    WHERE event_id = ?
");
$stmt->execute([$currentEventId]);
$summary = $stmt->fetch() ?: [
    'guests' => 0, 'people' => 0, 'amount_total' => 0,
    'amount_paid' => 0, 'amount_due' => 0,
    'count_paid' => 0, 'count_partial' => 0, 'count_unpaid' => 0,
    'count_checked' => 0,
];

// --- BREAKDOWN BY TICKET TYPE ---
$stmtType = $pdo->prepare("
    SELECT
        pricing,
        COUNT(*) AS guests,
        COALESCE(SUM(people_count),0) AS people,
        COALESCE(SUM(COALESCE(price_total,0)),0) AS amount_total,
        COALESCE(SUM(COALESCE(amount_paid,0)),0) AS amount_paid
    FROM users
    WHERE event_id = ?
    GROUP BY pricing
");
$stmtType->execute([$currentEventId]);
$byType = $stmtType->fetchAll();

// --- RECENT RSVPs ---
$stmtRsvp = $pdo->prepare("
    SELECT *
    FROM rsvps
    WHERE event_id = ?
    ORDER BY id DESC
    LIMIT 10
");
$stmtRsvp->execute([$currentEventId]);
$rsvps = $stmtRsvp->fetchAll();

// --- RECENT SMS LOGS (for guests in this event) ---
$stmtLogs = $pdo->prepare("
    SELECT s.*, u.name AS guest_name, u.phone AS guest_phone
    FROM sends s
    LEFT JOIN users u ON u.id = s.user_id
    WHERE s.channel LIKE 'sms%' AND (u.event_id = :event OR u.event_id IS NULL)
    ORDER BY s.id DESC
    LIMIT 10
");
$stmtLogs->execute(['event' => $currentEventId]);
$logs = $stmtLogs->fetchAll();

include __DIR__ . '/_layout_top.php';
?>

<style>
    .reports-grid {
        display:grid;
        grid-template-columns:repeat(auto-fit,minmax(230px,1fr));
        gap:10px;
        margin-bottom:14px;
    }
    .reports-card {
        background:linear-gradient(135deg,#0f172a,#020617);
        border-radius:14px;
        color:#e5e7eb;
        padding:10px 12px;
        border:1px solid rgba(148,163,184,.4);
        box-shadow:0 16px 40px rgba(15,23,42,.45);
        font-size:.8rem;
    }
    .reports-label {
        font-size:.72rem;
        text-transform:uppercase;
        letter-spacing:.12em;
        color:#9ca3af;
        margin-bottom:2px;
    }
    .reports-value {
        font-size:1.1rem;
        font-weight:600;
    }
    .reports-sub {
        font-size:.76rem;
        color:#9ca3af;
        margin-top:2px;
    }
    .table-clean {
        width:100%;
        border-collapse:collapse;
        font-size:.86rem;
    }
    .table-clean th,
    .table-clean td {
        padding:7px 8px;
        border-bottom:1px solid #e5e7eb;
        text-align:left;
        vertical-align:top;
    }
    .table-clean th {
        background:#f3f4f6;
        font-size:.78rem;
        text-transform:uppercase;
        letter-spacing:.06em;
        color:#6b7280;
    }
    .table-clean tbody tr:last-child td { border-bottom:none; }
    .reports-actions {
        display:flex;
        flex-wrap:wrap;
        gap:8px;
        margin-bottom:10px;
    }
</style>

<div>
    <h1 class="page-heading">Reports & Exports</h1>
    <p class="page-subtitle">
        Detailed overview for event:
        <strong><?= htmlspecialchars($currentEvent['name'] ?? '') ?></strong>
        <?php if (!empty($currentEvent['event_date'])): ?>
            (<?= htmlspecialchars($currentEvent['event_date']) ?>)
        <?php endif; ?>
    </p>
</div>

<div class="reports-grid">
    <div class="reports-card">
        <div class="reports-label">Guests</div>
        <div class="reports-value"><?= (int)$summary['guests'] ?></div>
        <div class="reports-sub"><?= (int)$summary['people'] ?> people capacity</div>
    </div>
    <div class="reports-card">
        <div class="reports-label">Amounts (TZS)</div>
        <div class="reports-value"><?= number_format($summary['amount_total'],0) ?></div>
        <div class="reports-sub">
            Expected · Paid <?= number_format($summary['amount_paid'],0) ?> ·
            Due <?= number_format($summary['amount_due'],0) ?>
        </div>
    </div>
    <div class="reports-card">
        <div class="reports-label">Payment status</div>
        <div class="reports-sub">
            <strong><?= (int)$summary['count_paid'] ?></strong> Paid ·
            <strong><?= (int)$summary['count_partial'] ?></strong> Partial ·
            <strong><?= (int)$summary['count_unpaid'] ?></strong> Unpaid
        </div>
        <div class="reports-sub" style="margin-top:4px;">
            Check-ins: <strong><?= (int)$summary['count_checked'] ?></strong>
        </div>
    </div>
    <div class="reports-card">
        <div class="reports-label">Exports</div>
        <div class="reports-sub">
            Download CSV files for guests, payments, and RSVPs.
        </div>
    </div>
</div>

<div class="card">
    <div class="card-title">Exports</div>
    <div class="reports-actions" style="margin-top:6px;">
        <a class="btn-action" href="export_guests_csv.php" target="_blank">Download Guests CSV</a>
        <a class="btn-action" href="export_payments_csv.php" target="_blank">Download Payments CSV</a>
        <a class="btn-action" href="export_rsvps_csv.php" target="_blank">Download RSVPs CSV</a>
    </div>
</div>

<div class="card">
    <div class="card-title">Breakdown by ticket type</div>
    <div style="overflow-x:auto;margin-top:8px;">
        <table class="table-clean">
            <thead>
            <tr>
                <th>Type</th>
                <th>Guests</th>
                <th>People</th>
                <th>Total (TZS)</th>
                <th>Paid (TZS)</th>
            </tr>
            </thead>
            <tbody>
            <?php if (!$byType): ?>
                <tr>
                    <td colspan="5" style="text-align:center;color:#9ca3af;padding:14px 8px;">
                        No guests for this event yet.
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($byType as $row): ?>
                    <tr>
                        <td><?= htmlspecialchars(ucfirst($row['pricing'] ?? 'N/A')) ?></td>
                        <td><?= (int)$row['guests'] ?></td>
                        <td><?= (int)$row['people'] ?></td>
                        <td><?= number_format($row['amount_total'],0) ?></td>
                        <td><?= number_format($row['amount_paid'],0) ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<div class="card">
    <div class="card-title">Recent RSVPs (last 10)</div>
    <div style="overflow-x:auto;margin-top:8px;">
        <table class="table-clean">
            <thead>
            <tr>
                <th>ID</th>
                <th>Guest</th>
                <th>Phone</th>
                <th>Message</th>
                <th>Created</th>
            </tr>
            </thead>
            <tbody>
            <?php if (!$rsvps): ?>
                <tr>
                    <td colspan="5" style="padding:14px 8px;text-align:center;color:#9ca3af;">
                        No RSVPs for this event yet.
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($rsvps as $r): ?>
                    <tr>
                        <td><?= (int)$r['id'] ?></td>
                        <td><?= htmlspecialchars($r['name']) ?></td>
                        <td><?= htmlspecialchars($r['phone']) ?></td>
                        <td style="max-width:260px;">
                            <div style="max-height:70px;overflow:auto;white-space:pre-wrap;">
                                <?= htmlspecialchars($r['message']) ?>
                            </div>
                        </td>
                        <td><?= htmlspecialchars($r['created_at']) ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<div class="card">
    <div class="card-title">Recent SMS activity (last 10)</div>
    <div style="overflow-x:auto;margin-top:8px;">
        <table class="table-clean">
            <thead>
            <tr>
                <th>ID</th>
                <th>Guest</th>
                <th>Channel</th>
                <th>Message</th>
                <th>Status</th>
                <th>Created</th>
            </tr>
            </thead>
            <tbody>
            <?php if (!$logs): ?>
                <tr>
                    <td colspan="6" style="padding:14px 8px;text-align:center;color:#9ca3af;">
                        No SMS logs yet for this event.
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($logs as $log): ?>
                    <tr>
                        <td><?= (int)$log['id'] ?></td>
                        <td>
                            <?php if ($log['guest_name']): ?>
                                <strong><?= htmlspecialchars($log['guest_name']) ?></strong><br>
                                <span class="pill-label"><?= htmlspecialchars($log['guest_phone']) ?></span>
                            <?php else: ?>
                                <span class="pill-label">Direct phone</span>
                            <?php endif; ?>
                        </td>
                        <td><?= htmlspecialchars($log['channel']) ?></td>
                        <td style="max-width:260px;">
                            <div style="max-height:70px;overflow:auto;white-space:pre-wrap;">
                                <?= htmlspecialchars($log['payload']) ?>
                            </div>
                        </td>
                        <td>
                            <?php if ($log['status'] === 'sent'): ?>
                                <span class="badge badge-paid">Sent</span>
                            <?php else: ?>
                                <span class="badge badge-unpaid"><?= htmlspecialchars($log['status']) ?></span>
                            <?php endif; ?>
                        </td>
                        <td><?= htmlspecialchars($log['created_at'] ?? '') ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include __DIR__ . '/_layout_bottom.php'; ?>