<?php
require_once __DIR__ . '/_auth.php';
require_once __DIR__ . '/../src/db.php';
require_once __DIR__ . '/../src/events.php';
require_once __DIR__ . '/../src/sms.php';

$pageTitle  = 'Bulk SMS';
$activeMenu = 'bulk_sms';

$currentEventId = getCurrentEventId();
$events = getAllEvents();
$currentEvent = null;
foreach ($events as $ev) {
    if ((int)$ev['id'] === $currentEventId) {
        $currentEvent = $ev;
        break;
    }
}

// Load guest list for current event (for single-send dropdown)
$stmtGuests = $pdo->prepare("SELECT id, name, phone, token FROM users WHERE event_id = ? ORDER BY name ASC");
$stmtGuests->execute([$currentEventId]);
$guestList = $stmtGuests->fetchAll();

// Load SMS templates
$stmtTpl = $pdo->query("SELECT * FROM sms_templates ORDER BY is_default DESC, id DESC");
$templates = $stmtTpl->fetchAll();

$msg = '';
$err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $segment        = $_POST['segment'] ?? 'all'; // all / paid / partial / unpaid / single
    $text           = trim($_POST['message'] ?? '');
    $templateId     = (int)($_POST['template_id'] ?? 0);

    // Optional guest selection
    $singleSelectId = (int)($_POST['single_guest_select'] ?? 0);
    $singleId       = (int)($_POST['single_guest_id'] ?? 0);
    $singleTel      = trim($_POST['single_phone'] ?? '');

    if ($singleSelectId > 0) {
        $singleId = $singleSelectId;
    }

    if (!$text) {
        $err = 'Message text is required.';
    } else {
        $targets = [];

        if ($segment === 'single') {
            if ($singleId > 0) {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE id=? AND event_id=? LIMIT 1");
                $stmt->execute([$singleId, $currentEventId]);
                if ($row = $stmt->fetch()) {
                    $targets[] = $row;
                } else {
                    $err = 'Guest not found for this event.';
                }
            } elseif ($singleTel !== '') {
                // pseudo guest for manual phone
                $targets[] = [
                    'id'       => null,
                    'event_id' => $currentEventId,
                    'name'     => 'Guest',
                    'phone'    => $singleTel,
                    'token'    => null,
                ];
            } else {
                $err = 'Select a guest or provide guest ID/phone for single send.';
            }
        } else {
            $sql = "SELECT * FROM users WHERE event_id = ?";
            $params = [$currentEventId];

            if ($segment === 'paid') {
                $sql .= " AND ((price_total IS NOT NULL AND amount_paid >= price_total AND price_total > 0) OR paid=1)";
            } elseif ($segment === 'partial') {
                $sql .= " AND price_total IS NOT NULL AND price_total > 0 AND amount_paid > 0 AND amount_paid < price_total";
            } elseif ($segment === 'unpaid') {
                $sql .= " AND ((price_total IS NOT NULL AND (amount_paid = 0 OR amount_paid IS NULL)) OR paid = 0)";
            }

            $sql .= " ORDER BY created_at ASC";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $targets = $stmt->fetchAll();
        }

        if (!$err && !$targets) {
            $err = 'No recipients found for this selection.';
        }

        if (!$err && $targets) {
            // use BASE_URL from env/sms.php if available
            global $BASE_URL;
            $baseUrl = rtrim($BASE_URL ?? ($_ENV['BASE_URL'] ?? ''), '/');
            if ($baseUrl === '') {
                // final fallback – your local public folder
                $baseUrl = 'http://localhost/tumasse/public';
            }

            $sentCount = 0;

            foreach ($targets as $g) {
                $phone = $g['phone'] ?? $singleTel;
                if (!$phone) {
                    continue;
                }

                // Personalize message
                $msgText = $text;

                if (!empty($g['token'])) {
                    // make sure link points to public guest page
                    $ticketUrl = $baseUrl . '/guest.php?token=' . $g['token'];
                    $msgText = str_replace(
                        ['{name}', '{link}'],
                        [$g['name'] ?? 'Mgeni', $ticketUrl],
                        $text
                    );
                } else {
                    $msgText = str_replace('{name}', $g['name'] ?? 'Mgeni', $text);
                    $msgText = str_replace('{link}', '', $msgText);
                }

                // *** NEW: use sms helper from sms.php ***
                $res = sms_send_to_guest($g, $msgText, 'sms_bulk');

                $status   = $res['status']   ?? 'failed';
                $response = $res['response'] ?? ($res['error'] ?? '');

                // keep existing simple logging for compatibility
                $pdo->prepare("INSERT INTO sends (user_id, channel, payload, status, response)
                               VALUES (?, 'sms_bulk', ?, ?, ?)")
                    ->execute([
                        $g['id'] ?? null,
                        $msgText,
                        $status,
                        is_array($response) ? json_encode($response) : (string)$response,
                    ]);

                if ($status === 'sent') {
                    $sentCount++;
                }
            }

            $msg = "Bulk SMS sent. Recipients: " . count($targets) . ", reported sent: $sentCount.";
        }
    }
}

include __DIR__ . '/_layout_top.php';
?>

<div>
    <h1 class="page-heading">Bulk SMS</h1>
    <p class="page-subtitle">
        Send SMS to guests of the current event:
        <strong><?= htmlspecialchars($currentEvent['name'] ?? '') ?></strong>.
        Use placeholders <code>{name}</code> and <code>{link}</code> to personalize the message.
    </p>
</div>

<div class="card" style="max-width:920px;">
    <?php if ($msg): ?>
        <div style="
            background:#e0f2fe;border:1px solid #bfdbfe;color:#1d4ed8;
            padding:8px 10px;border-radius:8px;font-size:.85rem;margin-bottom:12px;">
            <?= htmlspecialchars($msg) ?>
        </div>
    <?php endif; ?>
    <?php if ($err): ?>
        <div style="
            background:#fee2e2;border:1px solid #fecaca;color:#b91c1c;
            padding:8px 10px;border-radius:8px;font-size:.85rem;margin-bottom:12px;">
            <?= htmlspecialchars($err) ?>
        </div>
    <?php endif; ?>

    <form method="post" style="display:flex;flex-direction:column;gap:12px;">
        <div class="grid" style="grid-template-columns:repeat(auto-fit,minmax(240px,1fr));gap:10px;">
            <div>
                <label style="font-size:.85rem;">Recipient group</label><br>
                <select name="segment"
                        style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
                    <option value="all">All guests (this event)</option>
                    <option value="paid">Paid in full</option>
                    <option value="partial">Partial paid</option>
                    <option value="unpaid">Unpaid</option>
                    <option value="single">Single guest / phone</option>
                </select>
                <div style="font-size:.75rem;color:#6b7280;margin-top:2px;">
                    For <strong>Single</strong> choose a guest or type ID/phone below.
                </div>
            </div>

            <div>
                <label style="font-size:.85rem;">SMS template (optional)</label><br>
                <select name="template_id" id="template-select"
                        style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
                    <option value="">-- No template (write message) --</option>
                    <?php foreach ($templates as $tpl): ?>
                        <option value="<?= (int)$tpl['id'] ?>"
                                data-body="<?= htmlspecialchars($tpl['body'], ENT_QUOTES) ?>">
                            <?= $tpl['is_default'] ? '★ ' : '' ?><?= htmlspecialchars($tpl['name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <div style="font-size:.75rem;color:#6b7280;margin-top:2px;">
                    Selecting a template will load its text into the box below.
                </div>
            </div>
        </div>

        <div class="grid" style="grid-template-columns:repeat(auto-fit,minmax(260px,1fr));gap:10px;">
            <div>
                <label style="font-size:.85rem;">Select guest (optional, single send)</label><br>
                <select name="single_guest_select"
                        style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid #d5d8e5;">
                    <option value="">-- Choose guest from list --</option>
                    <?php foreach ($guestList as $g): ?>
                        <option value="<?= (int)$g['id'] ?>">
                            #<?= (int)$g['id'] ?> · <?= htmlspecialchars($g['name']) ?>
                            (<?= htmlspecialchars($g['phone']) ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div>
                <label style="font-size:.85rem;">Single guest ID (optional)</label><br>
                <input type="number" name="single_guest_id"
                       placeholder="e.g. 15"
                       style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid:#d5d8e5;">
            </div>

            <div>
                <label style="font-size:.85rem;">Single phone (optional)</label><br>
                <input type="text" name="single_phone"
                       placeholder="e.g. 0754xxxxxx"
                       style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid:#d5d8e5;">
            </div>
        </div>

        <div>
            <label style="font-size:.85rem;">Message</label><br>
            <textarea name="message" id="message-text" rows="6"
                      placeholder="Habari {name},&#10;Huu ni mwaliko wako kwa tukio letu. Bofya {link} kuona kadi yako."
                      style="width:100%;padding:7px 9px;border-radius:8px;border:1px solid:#d5d8e5;font-family:inherit;font-size:.9rem;"></textarea>
            <div style="font-size:.75rem;color:#6b7280;margin-top:2px;">
                Placeholders:
                <code>{name}</code> = guest name,
                <code>{link}</code> = personal invitation link.
            </div>
        </div>

        <!-- Live preview -->
        <div class="card" style="background:#0f172a;color:#e5e7eb;">
            <div style="font-size:.8rem;text-transform:uppercase;letter-spacing:.12em;color:#9ca3af;margin-bottom:4px;">
                Preview (example guest)
            </div>
            <div id="preview-box" style="font-size:.9rem;white-space:pre-wrap;">
                Habari {name},
                Huu ni mwaliko wako kwa tukio letu. Bofya {link} kuona kadi yako.
            </div>
        </div>

        <div style="margin-top:6px;display:flex;gap:10px;flex-wrap:wrap;">
            <button class="btn-action primary" type="submit">Send SMS</button>
            <a href="sms_templates.php" class="btn-action">Manage templates</a>
            <a href="sms_logs.php" class="btn-action">View logs</a>
            <a href="guests.php" class="btn-action">Back to Guests</a>
        </div>
    </form>
</div>

<script>
    const tplSelect = document.getElementById('template-select');
    const msgText   = document.getElementById('message-text');
    const preview   = document.getElementById('preview-box');

    function updatePreview() {
        let txt = msgText.value || '';
        txt = txt.replace(/\{name\}/g, 'Maheemba');
        txt = txt.replace(/\{link\}/g, 'https://example.com/your-ticket-link');
        preview.textContent = txt;
    }

    tplSelect.addEventListener('change', function () {
        const option = tplSelect.options[tplSelect.selectedIndex];
        const body = option.getAttribute('data-body') || '';
        if (body) {
            msgText.value = body;
            updatePreview();
        }
    });

    msgText.addEventListener('input', updatePreview);
    updatePreview();
</script>

<?php include __DIR__ . '/_layout_bottom.php'; ?>