<?php
// src/sms.php
//
// Central SMS helper for TUMESSE EVENTS
// - Uses Beem Africa HTTP API
// - Logs messages into `sends` table
// - Provides simple helper functions used by the admin UI

require_once __DIR__ . '/db.php';

// --- Load environment/config -------------------------------------------------
$env = [];
$envFile = __DIR__ . '/env.php';
if (file_exists($envFile)) {
    $env = require $envFile;
}

$BEEM_API_KEY    = $env['beem_api_key']    ?? '';
$BEEM_SECRET_KEY = $env['beem_secret_key'] ?? '';
$BEEM_SENDER_ID  = $env['beem_sender_id']  ?? 'TUMESSE';
$BASE_URL        = $env['base_url']        ?? 'https://apisms.beem.africa/v1/send';
$BEEM_BASE_URL   = $env['beem_base_url']        ?? 'https://apisms.beem.africa/v1/send';
$ENABLE_SMS      = $env['enable_sms']      ?? true;


/**
 * Normalise Tanzanian phone numbers for Beem (e.g. 0712... -> 255712...).
 */
function sms_normalize_phone(string $phone): string
{
    // Keep digits only
    $p = preg_replace('/\D+/', '', $phone ?? '');

    if ($p === '') {
        return '';
    }

    // Already 2557...
    if (preg_match('/^2557\d{7}$/', $p)) {
        return $p;
    }

    // Leading 0 -> 255
    if (preg_match('/^0[67]\d{8}$/', $p)) {
        return '255' . substr($p, 1);
    }

    // 7xxxxxxxx -> 2557...
    if (preg_match('/^[67]\d{8}$/', $p)) {
        return '255' . $p;
    }

    // Fallback: return as is
    return $p;
}

/**
 * Low-level function: actually send via Beem.
 * Returns associative array: ['success' => bool, 'status' => string, 'response' => string]
 */
function sms_beem_send_raw(string $phone, string $message): array
{
    global $BEEM_API_KEY, $BEEM_SECRET_KEY, $BEEM_SENDER_ID, $ENABLE_SMS;

    $result = [
        'success'  => false,
        'status'   => 'disabled',
        'response' => '',
    ];

    if (!$ENABLE_SMS) {
        $result['status'] = 'disabled';
        return $result;
    }

    $normalized = sms_normalize_phone($phone);
    if ($normalized === '') {
        $result['status'] = 'invalid_phone';
        return $result;
    }

    if ($BEEM_API_KEY === '' || $BEEM_SECRET_KEY === '') {
        $result['status'] = 'missing_keys';
        return $result;
    }

    $payload = [
        'source_addr'   => $BEEM_SENDER_ID,
        'encoding'      => 0,
        'schedule_time' => '',
        'message'       => [
            [
                'phonenumber' => $normalized,
                'content'     => $message,
            ],
        ],
    ];

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => 'https://apisms.beem.africa/v1/send',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Authorization: Basic ' . base64_encode($BEEM_API_KEY . ':' . $BEEM_SECRET_KEY),
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload),
        CURLOPT_TIMEOUT        => 30,
    ]);

    $response = curl_exec($ch);
    $err      = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $result['response'] = $response !== false ? $response : $err;

    if ($err) {
        $result['status'] = 'curl_error';
    } elseif ($httpCode >= 200 && $httpCode < 300) {
        $result['success'] = true;
        $result['status']  = 'sent';
    } else {
        $result['status'] = 'http_' . $httpCode;
    }

    return $result;
}

/**
 * Log SMS attempt into `sends` table.
 */
function sms_log_send(
    string $phone,
    string $message,
    string $channel,
    string $status,
    string $response = '',
    ?int $eventId = null,
    ?int $guestId = null
): void {
    global $pdo;

    try {
        $stmt = $pdo->prepare("
            INSERT INTO sends (event_id, guest_id, phone, channel, message, status, response_raw, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $eventId,
            $guestId,
            $phone,
            $channel,
            $message,
            $status,
            $response,
        ]);
    } catch (Throwable $e) {
        // do not break main flow
    }
}

/**
 * High-level helper: send one SMS and log it.
 * Returns the array from sms_beem_send_raw() + 'normalized_phone'.
 */
function sms_send_single(
    string $phone,
    string $message,
    string $channel = 'sms',
    ?int $eventId = null,
    ?int $guestId = null
): array {
    $normalized = sms_normalize_phone($phone);
    $result     = sms_beem_send_raw($phone, $message);

    sms_log_send(
        $normalized ?: $phone,
        $message,
        $channel,
        $result['status'],
        (string)($result['response'] ?? ''),
        $eventId,
        $guestId
    );

    $result['normalized_phone'] = $normalized;
    return $result;
}

/**
 * Convenience wrapper (older code may call this).
 */
function send_sms(string $phone, string $message, array $meta = []): array
{
    $eventId = $meta['event_id'] ?? null;
    $guestId = $meta['guest_id'] ?? null;
    $channel = $meta['channel']  ?? 'sms';

    return sms_send_single($phone, $message, $channel, $eventId, $guestId);
}

/**
 * Another helper used by bulk pages:
 * Accepts a full guest row as array, plus message text.
 */
function sms_send_to_guest(array $guest, string $message, string $channel = 'sms'): array
{
    $phone   = $guest['phone']    ?? '';
    $eventId = isset($guest['event_id']) ? (int)$guest['event_id'] : null;
    $guestId = isset($guest['id'])       ? (int)$guest['id']       : null;

    return sms_send_single($phone, $message, $channel, $eventId, $guestId);
}